/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	$Id: PGPPassphraseDialog.cpp,v 1.18 2001/01/04 20:12:54 wjb Exp $
____________________________________________________________________________*/

#include "stdio.h"
#include "stdlib.h"
#include "PGPui32.h"
#include "pgpDialogs.h"
#include "pgpPassphraseUtils.h"
#include "PGPCL.h"
#include "pgpSDKUILibPriv.h"
#include "SpaceEdit.h"
#include "InfoBar.h"

#define MAXDECRYPTIONNAMECHAR		36

// local globals
HHOOK				hhookKeyboard;
HHOOK				hhookMouse;

HHOOK				hhookCBT;
HHOOK				hhookGetMessage;
HHOOK				hhookMsgFilter;

// global variable structure for re-entrancy
typedef struct _GPP
{
	LPSTR				pszPassPhrase;
	LPSTR				pszPassPhraseConf;
	WNDPROC				wpOrigPhrase1Proc;  
	WNDPROC				wpOrigPhrase2Proc;  
	INT					iNextTabControl;
	BOOL				bHideText;
	HWND				hwndQuality;
	HWND				hwndMinQuality;
	PGPContextRef		context;
	HWND				hwndOptions;
	const CPGPPassphraseDialogOptions *options;
	HWND				hwndOptionsControl;
	int					NoAnimIndex;
	HICON				hInfoIcon;
	BOOL				bNoOK;
} GPP;

// Help IDs for passphrase dialogs
static DWORD aIds[] = 
{			
    IDC_HIDETYPING,		IDH_PGPCLPHRASE_HIDETYPING, 
	IDC_PHRASE1,		IDH_PGPCLPHRASE_PHRASE,
	IDC_PHRASE2,		IDH_PGPCLPHRASE_CONFIRMATION,
	IDC_SIGNKEYCOMBO,	IDH_PGPCLPHRASE_SIGNINGKEY,
	IDC_KEYLISTBOX,		IDH_PGPCLPHRASE_KEYLIST,
	IDC_PHRASEQUALITY,	IDH_PGPCLPHRASE_QUALITY,
	IDC_KEYNAME,		IDH_PGPCLPHRASE_KEYNAME,
	801,				IDH_TEXTOUTPUT, // Hardcoded in ClientLib
	804,				IDH_DETACHEDSIG, // Hardcoded in ClientLib
	807,				IDH_INPUTISTEXT, // Hardcoded in ClientLib
    0,0 
}; 

#define NOANIMNUM 8
#define NOANIMTIMER 444
#define NOANIMINTERVAL 40
static int NoAnim[NOANIMNUM]={-3,3,3,-3,-3,3,3,-3};

PGPError
PGPsdkUIErrorBox (
		HWND hWnd,
		PGPError error) 
{
	PGPError	err				= kPGPError_NoErr;
	CHAR		szMessage[512];
	char		StrRes[500];

	if (IsPGPError (error) && (error!=kPGPError_UserAbort)) 
	{
		PGPGetErrorString (error, sizeof(szMessage), szMessage);

		LoadString (gPGPsdkUILibInst, IDS_PGPERROR, StrRes, sizeof(StrRes));

		MessageBox (hWnd, szMessage, StrRes, 
			MB_OK|MB_ICONSTOP|MB_SETFOREGROUND);
	}

	return err;
}

// SetCapsLockMessageState shows or hides the caps lock message as needed.

void SetCapsLockMessageState(HWND hdlg)
{
	if (GetKeyState(VK_CAPITAL) & 1)
	{
		ShowWindow(GetDlgItem(hdlg,IDC_CAPSWARNING),SW_SHOW);
	}
	else
	{
		ShowWindow(GetDlgItem(hdlg,IDC_CAPSWARNING),SW_HIDE);
	}
}

// Help file is located in C:\Windows\System
void GetHelpDir(char *szHelp)
{
	GetSystemDirectory(szHelp,MAX_PATH);
	strcat(szHelp,"\\Pgp.chm");
}

//	________________________________
//
//  Hook procedure for WH_MOUSE hook
  
LRESULT CALLBACK 
MouseHookProc (
		INT		iCode, 
		WPARAM	wParam, 
		LPARAM	lParam) 
{ 
	MOUSEHOOKSTRUCT* pmhs;

    if (iCode >= 0) 
	{
		pmhs = (MOUSEHOOKSTRUCT*) lParam; 
		PGPGlobalRandomPoolMouseMoved ();
	}

	return 0;
} 

//	___________________________________
//
//  Hook procedure for WH_KEYBOARD hook 

 
LRESULT CALLBACK 
GetKBHookProc (
		INT		iCode, 
		WPARAM	wParam, 
		LPARAM	lParam) 
{ 
    if (iCode >= 0) 
	{
		// only use key presses, not releases
		if (!(lParam & 0x80000000))
			PGPGlobalRandomPoolAddKeystroke (wParam);
	}

	return 0;
} 
  
//	___________________________________
//
//  Hook procedure for WH_JOURNALRECORD hook 

 
LRESULT CALLBACK 
GenericHookProc (
		INT		iCode, 
		WPARAM	wParam, 
		LPARAM	lParam) 
{ 
	return 0;
} 

LRESULT CALLBACK 
CBTHookProc (
		INT		iCode, 
		WPARAM	wParam, 
		LPARAM	lParam) 
{ 
	if (iCode >= 0)
	{
		switch (iCode)
        {
			case HCBT_SETFOCUS:
				return CallNextHookEx (hhookCBT, iCode, wParam, lParam);
				break;
		}
	}

	return 0;
} 


void InitRandomKeyHook(HHOOK* phhookKeyboard, HHOOK* phhookMouse)
{
	DWORD dwPhraseThreadID;

	dwPhraseThreadID = GetCurrentThreadId ();

	// the mouse and keyboard hooks trap entropy
	*phhookMouse = SetWindowsHookEx (WH_MOUSE, 
		MouseHookProc, 
		NULL, dwPhraseThreadID);
	*phhookKeyboard = SetWindowsHookEx (WH_KEYBOARD, 
		GetKBHookProc, 
		NULL, dwPhraseThreadID);
}

void UninitRandomKeyHook(HHOOK hhookKeyboard, HHOOK hhookMouse)
{
	UnhookWindowsHookEx (hhookKeyboard);
	UnhookWindowsHookEx (hhookMouse);
}

void InstallSecurityHooks(void)
{
	DWORD dwPhraseThreadID;

	dwPhraseThreadID = GetCurrentThreadId ();

	// these are just to prevent sniffers from seeing these messages
	hhookCBT = SetWindowsHookEx (WH_CBT, 
		CBTHookProc, 
		NULL, dwPhraseThreadID);
	hhookGetMessage = SetWindowsHookEx (WH_GETMESSAGE, 
		GenericHookProc, 
		NULL, dwPhraseThreadID);
	hhookMsgFilter = SetWindowsHookEx (WH_MSGFILTER, 
		GenericHookProc, 
		NULL, dwPhraseThreadID);
}

void UninstallSecurityHooks(void)
{
	UnhookWindowsHookEx (hhookMsgFilter);
	UnhookWindowsHookEx (hhookGetMessage);
	UnhookWindowsHookEx (hhookCBT);
}

	PGPUInt32
PGPEstimatePassphraseQuality(const char *passphrase)
{
	return( pgpEstimatePassphraseQuality( passphrase ) );
}

//___________________________
//
// Secure memory allocation routines
//

VOID* 
secAlloc (PGPContextRef context, UINT uBytes) 
{
	PGPMemoryMgrRef	memmgr;

	memmgr =  PGPPeekContextMemoryMgr (context);
	return (PGPNewSecureData (memmgr, uBytes, 0));
}


VOID 
secFree (VOID* p) 
{
	if (p) {
		FillMemory ((char *)p,lstrlen((char *)p), '\0');
		PGPFreeData ((char *)p);
	}
}

void FreePassphrases(GPP *gpp)
{
	if(gpp->pszPassPhrase)
	{
		secFree(gpp->pszPassPhrase);
		gpp->pszPassPhrase=NULL;
	}

	if(gpp->pszPassPhraseConf)
	{
		secFree(gpp->pszPassPhraseConf);
		gpp->pszPassPhraseConf=NULL;
	}
}

void ClearPassphrases(HWND hDlg,GPP *gpp)
{
	HWND hwndPhrase1,hwndPhrase2;

	if(gpp->pszPassPhraseConf)
	{
		secFree(gpp->pszPassPhraseConf);
		gpp->pszPassPhraseConf=NULL;
	}

	hwndPhrase1=GetDlgItem(hDlg,IDC_PHRASE1);
	hwndPhrase2=GetDlgItem(hDlg,IDC_PHRASE2);

	SEWipeEditBox (hwndPhrase1);
	SEWipeEditBox (hwndPhrase2);

	SetFocus (hwndPhrase1);
}

//	___________________________________________________
//
//	Message box routine using string table resource IDs

LRESULT 
PGPsdkUIMessageBox (
		 HWND	hWnd, 
		 INT	iCaption, 
		 INT	iMessage,
		 ULONG	ulFlags) 
{
	CHAR szCaption [128];
	CHAR szMessage [256];

	LoadString (gPGPsdkUILibInst, iCaption, szCaption, sizeof(szCaption));
	LoadString (gPGPsdkUILibInst, iMessage, szMessage, sizeof(szMessage));

	ulFlags |= MB_SETFOREGROUND;
	return (MessageBox (hWnd, szMessage, szCaption, ulFlags));
}

//	____________________________
//
//  setup key display list box

BOOL 
AddKeySetToRecipientsTable (HWND hDlg, 
							PGPKeySetRef KeySet,
							CPGPDecryptionPassphraseDialogOptions *options) 
{
	BOOL			bAtLeastOneValidSecretKey	= FALSE;
	UINT			uUnknownKeys				= 0;
	PGPKeyListRef	KeyList;
	PGPKeyIterRef	KeyIter;
	PGPKeyDBObjRef	Key, SubKey;
	UINT			u, uIndex, uKeyBits, uAlgorithm;
	INT				iKeyDefault, iKeySelected;
	PGPBoolean		bSecret, bCanDecrypt;
	CHAR			szName[kPGPMaxUserIDSize];
	CHAR			sz[128];

	PGPOrderKeySet (KeySet, kPGPKeyOrdering_Validity, FALSE, &KeyList);
	PGPNewKeyIter (KeyList, &KeyIter);

	iKeySelected = -1;
	iKeyDefault = -1;

	PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Key, &Key);
	while (Key) {
		PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsSecret, &bSecret);
		PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_CanDecrypt, &bCanDecrypt);
		PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_AlgorithmID, (PGPInt32 *)&uAlgorithm);

		if (bSecret && bCanDecrypt) bAtLeastOneValidSecretKey = TRUE;

		// get name on key
		PGPGetPrimaryUserIDName(Key, szName, sizeof(szName), &u);
		if (u > MAXDECRYPTIONNAMECHAR) {
			u = MAXDECRYPTIONNAMECHAR;
			lstrcat (szName, "...");
		}
		else 
			szName[u] = '\0';

		// append key type / size info to name
		lstrcat (szName, " (");

		switch (uAlgorithm) 
		{
			case kPGPPublicKeyAlgorithm_RSA :
				LoadString (gPGPsdkUILibInst, IDS_RSA, sz, sizeof(sz));
				lstrcat (szName, sz);
				lstrcat (szName, "/");
				PGPGetKeyDBObjNumericProperty (Key, kPGPKeyProperty_Bits, (int *)&uKeyBits);
				wsprintf (sz, "%i", uKeyBits);
				lstrcat (szName, sz);
				break;

			case kPGPPublicKeyAlgorithm_DSA :
				LoadString (gPGPsdkUILibInst, IDS_DH, sz, sizeof(sz));
				lstrcat (szName, sz);
				lstrcat (szName, "/");
				PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_SubKey, &SubKey);
				if (SubKey) {
					PGPGetKeyDBObjNumericProperty(SubKey,
						kPGPSubKeyProperty_Bits, (PGPInt32 *)&uKeyBits);

					wsprintf (sz, "%i", uKeyBits);
					lstrcat (szName, sz);
				}
				else lstrcat (szName, "???");
				break;

			default :
				LoadString (gPGPsdkUILibInst, IDS_UNKNOWN, sz, sizeof(sz));
				lstrcat (szName, sz);
				lstrcat (szName, "/");
				lstrcat (szName, sz);
				break;
		}
		lstrcat (szName, ")");

		uIndex = SendDlgItemMessage (hDlg, IDC_KEYLISTBOX, 
							LB_ADDSTRING, 0, (LPARAM)szName);
		PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Key, &Key);
	}
	PGPFreeKeyIter (KeyIter);
	PGPFreeKeyList (KeyList);

	return bAtLeastOneValidSecretKey;
}

BOOL 
InitEncryptedToKeyListBox (HWND hDlg, CPGPDecryptionPassphraseDialogOptions *options) 
{
	BOOL bAtLeastOneValidSecretKey;

	SendMessage(GetDlgItem(hDlg, IDC_KEYLISTBOX),LB_RESETCONTENT,0,0);

	if(PGPKeySetRefIsValid( options->mKeySet ))
		bAtLeastOneValidSecretKey = AddKeySetToRecipientsTable( hDlg,options->mKeySet,options );

	if(options->mNewKeys!=NULL)
		if(PGPKeyDBRefIsValid( *(options->mNewKeys) ))
		{
			PGPKeySetRef keyset;
			PGPError err;
			err = PGPNewKeySet(*(options->mNewKeys), &keyset);
			if (err != kPGPError_NoErr) {
				AddKeySetToRecipientsTable( hDlg, keyset, options );
				PGPFreeKeySet(keyset);
			}
		}

	if( IsntNull( options->mMissingKeyIDList ) )
	{
		char MsgTxt[255];
		char StrRes[500];

		LoadString (gPGPsdkUILibInst, IDS_NUMUNKNOWNKEYS, StrRes, sizeof(StrRes));
	
		sprintf(MsgTxt,StrRes,options->mMissingKeyIDCount);

		SendDlgItemMessage (hDlg, IDC_KEYLISTBOX, LB_INSERTSTRING, -1, 
			(LPARAM)MsgTxt);
	}

	if(!bAtLeastOneValidSecretKey)
	{
		EnableWindow (GetDlgItem (hDlg, IDC_PHRASE1), FALSE);
		ShowWindow (GetDlgItem (hDlg, IDC_PHRASE1), SW_HIDE);
		EnableWindow (GetDlgItem (hDlg, IDC_HIDETYPING), FALSE);
		ShowWindow (GetDlgItem (hDlg, IDC_HIDETYPING), SW_HIDE);
		ShowWindow (GetDlgItem (hDlg, IDOK), SW_HIDE);
		EnableWindow (GetDlgItem (hDlg, IDOK), FALSE);
		ShowWindow (GetDlgItem (hDlg, IDC_PROMPTSTRING), SW_HIDE);
		ShowWindow (GetDlgItem (hDlg, IDC_CANNOTDECRYPTTEXT),SW_SHOW);
	}
	else
	{
		EnableWindow (GetDlgItem (hDlg, IDC_PHRASE1), TRUE);
		ShowWindow (GetDlgItem (hDlg, IDC_PHRASE1), SW_SHOW);
		EnableWindow (GetDlgItem (hDlg, IDC_HIDETYPING), TRUE);
		ShowWindow (GetDlgItem (hDlg, IDC_HIDETYPING), SW_SHOW);
		ShowWindow (GetDlgItem (hDlg, IDOK), SW_SHOW);
		EnableWindow (GetDlgItem (hDlg, IDOK), TRUE);
		ShowWindow (GetDlgItem (hDlg, IDC_PROMPTSTRING), SW_SHOW);
		ShowWindow (GetDlgItem (hDlg, IDC_CANNOTDECRYPTTEXT),SW_HIDE);
	}

	return bAtLeastOneValidSecretKey;
}

//	____________________________
//
//  Truncate text string

VOID
TruncateKeyText (HDC	hdc, 
				 LPSTR	pszOrig, 
				 LPSTR	szInfo, 
				 INT	iXmax, 
				 LPSTR	pszTrunc) 
{
	SIZE	s;
	INT		l, iW;

	GetTextExtentPoint32 (hdc, szInfo, lstrlen (szInfo), &s);
	iXmax -= s.cx;

	if (iXmax <= 0) {
		lstrcpy (pszTrunc, "");
		return;
	}

	lstrcpy (pszTrunc, pszOrig);
	GetTextExtentPoint32 (hdc, pszOrig, lstrlen (pszOrig), &s);
	iW = s.cx + 4;

	l = lstrlen (pszOrig);
	if (l < 3) {
		if (iW > iXmax) lstrcpy (pszTrunc, "");
	}
	else {
		l = lstrlen (pszOrig) - 3;
		while ((iW > iXmax) && (l >= 0)) {
			lstrcpy (&pszTrunc[l], "...");
			GetTextExtentPoint32 (hdc, pszTrunc, lstrlen (pszTrunc), &s);
			iW = s.cx + 4;
			l--;
		}	
		if (l < 0) lstrcpy (pszTrunc, "");
	}

	lstrcat (pszTrunc, szInfo);
	
}

void GetKeyString(HDC hdc,INT iComboWidth,PGPKeyDBObjRef Key,char *szNameFinal)
{
	CHAR			sz1[32],sz2[32];
	CHAR			szName[kPGPMaxUserIDSize];
	PGPUInt32		uAlgorithm,uKeyBits;
	UINT			u;

	PGPGetKeyDBObjNumericProperty (Key, kPGPKeyProperty_AlgorithmID, (int *)&uAlgorithm);

	// get key type / size info to append to name
	lstrcpy (sz2, "   (");
	switch (uAlgorithm) 
	{
		case kPGPPublicKeyAlgorithm_RSA :
			LoadString (gPGPsdkUILibInst, IDS_RSA, sz1, sizeof(sz1));
			lstrcat (sz2, sz1);
			lstrcat (sz2, "/");
			PGPGetKeyDBObjNumericProperty (Key, kPGPKeyProperty_Bits, (int *)&uKeyBits);
			wsprintf (sz1, "%i", uKeyBits);
			lstrcat (sz2, sz1);
			break;

		case kPGPPublicKeyAlgorithm_DSA :
			LoadString (gPGPsdkUILibInst, IDS_DSS, sz1, sizeof(sz1));
			lstrcat (sz2, sz1);
			lstrcat (sz2, "/");
			PGPGetKeyDBObjNumericProperty (Key, kPGPKeyProperty_Bits, (int *)&uKeyBits);
			wsprintf (sz1, "%i", uKeyBits);
			lstrcat (sz2, sz1);
			break;

		default :
			LoadString (gPGPsdkUILibInst, IDS_UNKNOWN, sz1, sizeof(sz1));
			lstrcat (sz2, sz1);
			lstrcat (sz2, "/");
			lstrcat (sz2, sz1);
			break;
	}
		
	lstrcat (sz2, ")");

	// get name on key
	PGPGetPrimaryUserIDName(Key, szName, sizeof(szName), &u);
	TruncateKeyText (hdc, szName, sz2, iComboWidth, szNameFinal);
}


//	____________________________
//
//  setup keyselection combo box

BOOL 
InitSigningKeyComboBox (HWND hDlg, CPGPKeySetPassphraseDialogOptions *options) 
{
	PGPKeyListRef	KeyList;
	PGPKeyIterRef	KeyIter;
	PGPKeyDBObjRef	Key;
	UINT			uIndex;
	INT				iKeyDefault, iKeySelected;
	PGPBoolean		bSecret, bRevoked, bExpired, bCanSign;
	BOOL			bAtLeastOneSecretKey;
	CHAR			szNameFinal[kPGPMaxUserIDSize];
	HDC				hdc;
	RECT			rc;
	INT				iComboWidth;
	HFONT			hFont;

	PGPOrderKeySet (options->mKeySet, kPGPKeyOrdering_Validity, FALSE, &KeyList);
	PGPNewKeyIter (KeyList, &KeyIter);

	SendDlgItemMessage (hDlg, IDC_SIGNKEYCOMBO, CB_GETDROPPEDCONTROLRECT,
						0, (LPARAM)&rc);
	iComboWidth = rc.right-rc.left - GetSystemMetrics (SM_CXVSCROLL);
	hdc = GetDC (GetDlgItem (hDlg, IDC_SIGNKEYCOMBO));
	hFont = (HFONT)GetStockObject (DEFAULT_GUI_FONT);
	SelectObject (hdc, hFont);

	iKeySelected = -1;
	iKeyDefault = 0;

	bAtLeastOneSecretKey = FALSE;

	PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Key, &Key);
	while (Key) {
		PGPGetKeyDBObjBooleanProperty (Key, kPGPKeyProperty_IsSecret, &bSecret);
		if (bSecret) {
			PGPGetKeyDBObjBooleanProperty (Key, kPGPKeyProperty_IsRevoked, (unsigned char *)&bRevoked);
			PGPGetKeyDBObjBooleanProperty (Key, kPGPKeyProperty_IsExpired, (unsigned char *)&bExpired);
			PGPGetKeyDBObjBooleanProperty (Key, kPGPKeyProperty_CanSign, (unsigned char *)&bCanSign);
			if (!bRevoked && !bExpired && bCanSign) {
				bAtLeastOneSecretKey = TRUE;

				GetKeyString(hdc,iComboWidth,Key,szNameFinal);

				uIndex = SendDlgItemMessage (hDlg, IDC_SIGNKEYCOMBO, 
								CB_ADDSTRING, 0, (LPARAM)szNameFinal);
				if (uIndex != CB_ERR) {
					SendDlgItemMessage (hDlg, IDC_SIGNKEYCOMBO, 
								CB_SETITEMDATA, uIndex, (LPARAM)Key);
					if (options->mDefaultKey) {
						if (Key == options->mDefaultKey) 
							iKeySelected = uIndex;
					}
					if (Key == options->mDefaultPrivateKey)
						iKeyDefault = uIndex;
				}
			}
		}
		PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Key, &Key);
	}
	PGPFreeKeyIter (KeyIter);
	PGPFreeKeyList (KeyList);

	if (iKeySelected == -1) iKeySelected = iKeyDefault;
	SendDlgItemMessage (hDlg, IDC_SIGNKEYCOMBO, CB_SETCURSEL, 
							iKeySelected, 0);

	ReleaseDC (GetDlgItem (hDlg, IDC_SIGNKEYCOMBO), hdc);

	return (bAtLeastOneSecretKey);

}

BOOL PassphraseLengthAndQualityOK(HWND hwnd,
								  CPGPPassphraseDialogOptions *options,
								  char *Passphrase)
{
	if(options->mMinPassphraseLength!=0)
	{
		if(strlen(Passphrase)<options->mMinPassphraseLength)
		{
			PGPsdkUIMessageBox (hwnd,
				IDS_PGPERROR,IDS_PASSNOTLONG,
				MB_OK|MB_ICONSTOP);

			return FALSE;
		}
	}

	if(options->mMinPassphraseQuality!=0)
	{
		if(PGPEstimatePassphraseQuality(Passphrase)<options->mMinPassphraseQuality)
		{
			PGPsdkUIMessageBox (hwnd,
				IDS_PGPERROR,IDS_PASSNOTQUALITY,
				MB_OK|MB_ICONSTOP);

			return FALSE;
		}
	}

	return TRUE;
}


//	______________________________________________________________
//
//  Enable or disable passphrase edit box on basis of selected key

VOID 
EnablePassPhraseControl (HWND hDlg, PGPContextRef context) 
{
	PGPKeyDBObjRef	Key;
	PGPBoolean	bNeedsPhrase,bShared;
	INT			i;

	i = SendDlgItemMessage (hDlg, IDC_SIGNKEYCOMBO, CB_GETCURSEL, 0, 0);
	if (i != CB_ERR) 
	{					
		Key = (PGPKeyDBObjRef)SendDlgItemMessage (hDlg, IDC_SIGNKEYCOMBO, 
											CB_GETITEMDATA, i, 0);
		if (Key) 
		{
			PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_NeedsPassphrase, &bNeedsPhrase);
			PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsSecretShared, &bShared);

			if (bNeedsPhrase) 
			{
				if (PGPPassphraseIsValid (Key, PGPOLastOption (context)))
				{
					EnableWindow (GetDlgItem (hDlg, IDC_PHRASE1), FALSE);
					ShowWindow (GetDlgItem (hDlg, IDC_PHRASE1), SW_HIDE);
					EnableWindow (GetDlgItem (hDlg, IDC_HIDETYPING), FALSE);
					ShowWindow (GetDlgItem (hDlg, IDC_HIDETYPING), SW_HIDE);
					ShowWindow (GetDlgItem (hDlg, IDC_PROMPTSTRING), SW_HIDE);
					ShowWindow (GetDlgItem (hDlg, IDC_SPLITINFO),SW_HIDE);
					ShowWindow (GetDlgItem (hDlg, IDC_CACHEINFO),SW_SHOW);
					ShowWindow (GetDlgItem (hDlg, IDC_NOPHRASEINFO),SW_HIDE);
					SetFocus (GetDlgItem (hDlg, IDOK));
				}
				else if (bShared)
				{
					EnableWindow (GetDlgItem (hDlg, IDC_PHRASE1), FALSE);
					ShowWindow (GetDlgItem (hDlg, IDC_PHRASE1), SW_HIDE);
					EnableWindow (GetDlgItem (hDlg, IDC_HIDETYPING), FALSE);
					ShowWindow (GetDlgItem (hDlg, IDC_HIDETYPING), SW_HIDE);
					ShowWindow (GetDlgItem (hDlg, IDC_PROMPTSTRING), SW_HIDE);
					ShowWindow (GetDlgItem (hDlg, IDC_SPLITINFO),SW_SHOW);
					ShowWindow (GetDlgItem (hDlg, IDC_CACHEINFO),SW_HIDE);
					ShowWindow (GetDlgItem (hDlg, IDC_NOPHRASEINFO),SW_HIDE);
					SetFocus (GetDlgItem (hDlg, IDOK));
				}
				else
				{
					SetDlgItemText (hDlg, IDC_PHRASE1, "");
					EnableWindow (GetDlgItem (hDlg, IDC_PHRASE1), TRUE);
					ShowWindow (GetDlgItem (hDlg, IDC_PHRASE1), SW_SHOW);
					EnableWindow (GetDlgItem (hDlg, IDC_HIDETYPING), TRUE);
					ShowWindow (GetDlgItem (hDlg, IDC_HIDETYPING), SW_SHOW);
					ShowWindow (GetDlgItem (hDlg, IDC_PROMPTSTRING), SW_SHOW);
					ShowWindow (GetDlgItem (hDlg, IDC_SPLITINFO),SW_HIDE);
					ShowWindow (GetDlgItem (hDlg, IDC_CACHEINFO),SW_HIDE);
					ShowWindow (GetDlgItem (hDlg, IDC_NOPHRASEINFO),SW_HIDE);
					SetFocus (GetDlgItem (hDlg, IDC_PHRASE1));
				}
			}
			else
			{
				EnableWindow (GetDlgItem (hDlg, IDC_PHRASE1), FALSE);
				ShowWindow (GetDlgItem (hDlg, IDC_PHRASE1), SW_HIDE);
				EnableWindow (GetDlgItem (hDlg, IDC_HIDETYPING), FALSE);
				ShowWindow (GetDlgItem (hDlg, IDC_HIDETYPING), SW_HIDE);
				ShowWindow (GetDlgItem (hDlg, IDC_PROMPTSTRING), SW_HIDE);
				ShowWindow (GetDlgItem (hDlg, IDC_SPLITINFO),SW_HIDE);
				ShowWindow (GetDlgItem (hDlg, IDC_CACHEINFO),SW_HIDE);
				ShowWindow (GetDlgItem (hDlg, IDC_NOPHRASEINFO),SW_SHOW);
				SetFocus (GetDlgItem (hDlg, IDOK));
			}
		}
	}
}

void ShiftDialog(HWND hdlg,int delta)
{
	RECT rc;

	GetWindowRect (hdlg, &rc);
	rc.left=rc.left+delta;

	SetWindowPos (hdlg, NULL,
		rc.left,rc.top,
		0, 0, SWP_NOSIZE | SWP_NOZORDER);
}

void DoBadPassphraseShake(HWND hdlg,GPP *gpp)
{
	if(gpp->NoAnimIndex==0)
	{
		ShiftDialog(hdlg,NoAnim[gpp->NoAnimIndex]);

		gpp->NoAnimIndex++;

		SetTimer (hdlg, NOANIMTIMER, NOANIMINTERVAL, NULL);
	}
}

//	____________________________
//
//  search keys for matching phrase

PGPError 
ValidateSigningPhrase (HWND hDlg, 
				GPP *gpp, 
				LPSTR pszPhrase, 
				PGPKeyDBObjRef key) 
{
	CHAR	szName[kPGPMaxUserIDSize];
	CHAR	sz[128];
	CHAR	sz2[kPGPMaxUserIDSize + 128];
	PGPSize	size;
	CPGPSigningPassphraseDialogOptions *options;
	HWND hwndStatic;

	options=(CPGPSigningPassphraseDialogOptions *)gpp->options;

	// does phrase match selected key ?
	if (PGPPassphraseIsValid (key, 
			PGPOPassphrase (gpp->context, pszPhrase),
				options->mCache ?
					PGPOCachePassphrase(gpp->context,
						options->mCacheTimeout, options->mCacheGlobal ) :
					PGPONullOption( gpp->context ),
			PGPOLastOption (gpp->context))) {
		*(options->mPassphraseKeyPtr) = key;
		return kPGPError_NoErr;
	}

	if(options->mFindMatchingKey)
	{
		// does phrase match any private key ?
		key=GetKeyForPassphrase(options->mKeySet,pszPhrase,TRUE);

		if (key!=NULL) 
		{
			// ask user to use other key
			LoadString (gPGPsdkUILibInst, IDS_FOUNDMATCHFORPHRASE, sz, sizeof(sz));
			PGPGetPrimaryUserIDName(key, szName, sizeof(szName), &size);
			szName[size] = '\0';

			wsprintf (sz2, sz, szName);
			LoadString (gPGPsdkUILibInst, IDS_PGP, sz, sizeof(sz));
			if (MessageBox (hDlg, sz2, sz, MB_ICONQUESTION|MB_YESNO) == IDYES) 
			{
				*(options->mPassphraseKeyPtr) = key;
				return kPGPError_NoErr;
			}
			return kPGPError_BadPassphrase;
		}
	}

	// phrase doesn't match any key
//	PGPsdkUIMessageBox (hDlg,
//		IDS_PGPERROR,IDS_BADPASSREENTER,
//		MB_OK|MB_ICONSTOP);
	DoBadPassphraseShake(hDlg,gpp);

	hwndStatic=GetDlgItem(hDlg,IDC_PROMPTSTRING);

	LoadString (gPGPsdkUILibInst, IDS_WRONGPHRASE, sz, sizeof (sz));
	SetWindowText(hwndStatic,sz);

	IBSetHighlightMode(hwndStatic,FALSE);

	return kPGPError_BadPassphrase;

}

VOID
SpaceEditCallback(HWND hwndEdit,void *pUserValue)
{
	GPP *gpp;
	BOOL OKactive,bShowCapsWarning;
	DWORD PassLen,PassQual;
	HWND hdlg;
	HWND hwndPhrase1,hwndPhrase2;

	hdlg=GetParent(hwndEdit);
	hwndPhrase1=GetDlgItem(hdlg,IDC_PHRASE1);
	hwndPhrase2=GetDlgItem(hdlg,IDC_PHRASE2);

	gpp=(GPP *)pUserValue;

	OKactive=TRUE;

	PassLen=SEGetTextLength(hwndEdit);
	PassQual=SEGetTextQuality(hwndEdit);
	bShowCapsWarning=SEGetShowWarning(hwndEdit);

	if(bShowCapsWarning)
	{
		ShowWindow(GetDlgItem(hdlg,IDC_CAPSWARNING),SW_SHOW);
	}
	else
	{
		ShowWindow(GetDlgItem(hdlg,IDC_CAPSWARNING),SW_HIDE);
	}

	if(hwndPhrase1==hwndEdit)
	{
		/*
		if(gpp->options->mMinPassphraseLength!=0)
		{
			if(PassLen<gpp->options->mMinPassphraseLength)
			{
				OKactive=FALSE;
			}
		}
		*/

		// Check Quality for setting OK to TRUE
		if(gpp->options->mMinPassphraseQuality!=0)
		{
			if(PassQual<gpp->options->mMinPassphraseQuality)
			{
				OKactive=FALSE;
			}
		}
	
		if(gpp->hwndQuality)
		{
			// Set new quality
			SendMessage (gpp->hwndQuality, PBM_SETPOS, PassQual, 0);
		}

		// Enable/disable OK
		if(!gpp->bNoOK)
		{
			EnableWindow (GetDlgItem (hdlg, IDOK), OKactive);
		}
	}
}

#define DIALOGTIMEOUTTIMER 1972

BOOL CALLBACK 
DoCommonCalls (
		HWND	hDlg, 
		UINT	uMsg, 
		WPARAM	wParam,
		LPARAM	lParam) 
{
	GPP *gpp;

	switch (uMsg)
	{
		case WM_INITDIALOG:
		{
			CPGPPassphraseDialogOptions *options;

			SetWindowLong (hDlg, GWL_USERDATA, lParam);
			gpp=(GPP *)lParam;

			options = (CPGPPassphraseDialogOptions *)gpp->options;

			if(options->mWindowTitle)
				SetWindowText(hDlg,options->mWindowTitle);

			if(options->mPrompt)
				SetDlgItemText (hDlg, IDC_PROMPTSTRING, options->mPrompt);

			gpp->bHideText = TRUE;
			CheckDlgButton (hDlg, IDC_HIDETYPING, BST_CHECKED);

			SetCapsLockMessageState(hDlg);

			gpp->hInfoIcon=LoadIcon(gPGPsdkUILibInst,MAKEINTRESOURCE(IDI_INFOICON));
			IBInit(GetDlgItem(hDlg,IDC_CAPSWARNING),gpp->hInfoIcon,FALSE);
			IBInit(GetDlgItem(hDlg,IDC_PROMPTSTRING),gpp->hInfoIcon,TRUE);

			if (options->mDialogTimeout!=0) 
			{
				SetTimer (hDlg, DIALOGTIMEOUTTIMER, 
					// minutes * 60 seconds/minute * 1000 u/second
					options->mDialogTimeout*60*1000, NULL);
			}

			SetForegroundWindow (hDlg);

			// Force focus to passphrase box
			SetFocus(GetDlgItem(hDlg, IDC_PHRASE1));
			break;
		}

		case WM_TIMER:
		{
			gpp=(GPP *)GetWindowLong (hDlg, GWL_USERDATA);

			switch(wParam)
			{
				case NOANIMTIMER:
				{
					if(gpp->NoAnimIndex!=NOANIMNUM)
					{
						ShiftDialog(hDlg,NoAnim[gpp->NoAnimIndex]);

						gpp->NoAnimIndex++;
					}
					else
					{
						KillTimer(hDlg,NOANIMTIMER);
						gpp->NoAnimIndex=0;
					}
					break;
				}

				case DIALOGTIMEOUTTIMER:
				{
					EndDialog (hDlg, kPGPError_UserAbort);
					break;
				}

				break;
			}
		}

		case WM_KEYUP:
		{
			SetCapsLockMessageState(hDlg);
			break;
		}

		case WM_HELP :
		{
			LPHELPINFO	phi = (LPHELPINFO)lParam;

			if (phi->iCtrlId != -1)
			{
				char szHelpFile[MAX_PATH+1];

				GetHelpDir(szHelpFile);
				strcat(szHelpFile, POPUPTEXTFILE);

				HtmlHelp ((HWND)phi->hItemHandle, szHelpFile, 
						HH_TP_HELP_WM_HELP, (DWORD)aIds);
			}
			return TRUE;
		}

		case WM_CONTEXTMENU :
		{
			if ((HWND)wParam != hDlg)
			{
				char szHelpFile[MAX_PATH+1];

				GetHelpDir(szHelpFile);
				strcat(szHelpFile, POPUPTEXTFILE);

				HtmlHelp ((HWND)wParam, szHelpFile, 
						HH_TP_HELP_CONTEXTMENU, (DWORD)aIds); 
			}
			return TRUE;
		}

		case WM_QUIT:
		case WM_CLOSE:
		case WM_DESTROY: 
		{
			HWND hwndPhrase1,hwndPhrase2;
			HWND hwndCaps,hwndPrompt;

			gpp=(GPP *)GetWindowLong (hDlg, GWL_USERDATA);

			ClearPassphrases(hDlg,gpp);

			hwndPhrase1=GetDlgItem(hDlg, IDC_PHRASE1);
			hwndPhrase2=GetDlgItem(hDlg, IDC_PHRASE2);

			if(hwndPhrase1)
				SEDestroy(hwndPhrase1);

			if(hwndPhrase2)
				SEDestroy(hwndPhrase2);

			hwndCaps=GetDlgItem(hDlg, IDC_CAPSWARNING);
			hwndPrompt=GetDlgItem(hDlg, IDC_PROMPTSTRING);

			if(hwndCaps)
				IBDestroy(hwndCaps);

			if(hwndPrompt)
				IBDestroy(hwndPrompt);

			DeleteObject(gpp->hInfoIcon);

			EndDialog(hDlg,kPGPError_UserAbort);
			break;
		}

		case WM_CTLCOLOREDIT:
		{
			HWND hwndPhrase1,hwndPhrase2;

			gpp=(GPP *)GetWindowLong (hDlg, GWL_USERDATA);

			hwndPhrase1=GetDlgItem(hDlg, IDC_PHRASE1);
			hwndPhrase2=GetDlgItem(hDlg, IDC_PHRASE2);

			if(lParam==0)
				break;

			if (((HWND)lParam == hwndPhrase1) ||
				((HWND)lParam == hwndPhrase2)) 
			{
				SetBkColor ((HDC)wParam, GetSysColor (COLOR_WINDOW));

				SetTextColor ((HDC)wParam, 
							  GetSysColor (COLOR_WINDOWTEXT));
				return (BOOL)CreateSolidBrush (GetSysColor (COLOR_WINDOW));
			}
			break;
		}

		case WM_COMMAND:
		{
			gpp=(GPP *)GetWindowLong (hDlg, GWL_USERDATA);

			switch(LOWORD (wParam)) 
			{
				case IDCANCEL:
					EndDialog (hDlg, kPGPError_UserAbort);
					break;

				case IDC_HIDETYPING :
				{
					HWND hwndPhrase1,hwndPhrase2;
	
					hwndPhrase1=GetDlgItem(hDlg, IDC_PHRASE1);
					hwndPhrase2=GetDlgItem(hDlg, IDC_PHRASE2);

					if (IsDlgButtonChecked (hDlg, IDC_HIDETYPING)
							== BST_CHECKED) 
						gpp->bHideText = TRUE;
					else 
						gpp->bHideText = FALSE;

					if(hwndPhrase1)
						SEChangeHideTyping (hwndPhrase1, gpp->bHideText);

					if(hwndPhrase2)
						SEChangeHideTyping (hwndPhrase2, gpp->bHideText);
					break;
				}
			}
			break;	
		}
	}

	return FALSE;
}

HWND DisplayOptions(HWND hDlg,PGPOptionListRef options,int y)
{
	if(options==NULL)
	{
		return NULL;
	}
	else
	{
		RECT rc;

		GetWindowRect(hDlg,&rc);

		SetWindowPos (hDlg, 
			NULL, 0, 0,
			rc.right-rc.left,rc.bottom-rc.top+60, //45,
				SWP_NOMOVE|SWP_NOZORDER);

		return CreateOptionsControl(hDlg,
			options,
			10,y,
			200,86);//71);
	}
}

// ****************************************************************************
// ****************************************************************************

//	________________________
//
//  Dialog Message procedure

BOOL CALLBACK 
pgpPassphraseDlgProc (
		HWND	hDlg, 
		UINT	uMsg, 
		WPARAM	wParam,
		LPARAM	lParam) 
{
	CPGPPassphraseDialogOptions *options;
	GPP				*gpp;
	INT				i;
	BOOL			Common;

	Common=DoCommonCalls (hDlg,uMsg,wParam,lParam); 

	if(Common)
		return Common;

	switch (uMsg) 
	{
		case WM_INITDIALOG:
		{
			gpp=(GPP *)GetWindowLong (hDlg, GWL_USERDATA);
			options = (CPGPPassphraseDialogOptions *)
				gpp->options;
	
			gpp->iNextTabControl = IDOK;

			SEInit(gpp->context,GetDlgItem(hDlg, IDC_PHRASE1),TRUE,
				SpaceEditCallback,(void *)gpp);

			gpp->hwndOptionsControl=DisplayOptions(hDlg,options->mDialogOptions,107);
			return FALSE;
		}

		case WM_COMMAND:
		{
			gpp=(GPP *)GetWindowLong (hDlg, GWL_USERDATA);
			options = (CPGPPassphraseDialogOptions *)
				gpp->options;
	
			switch(LOWORD (wParam)) 
			{
				case IDOK: 
				{
					FreePassphrases(gpp);

					i=SEGetTextLength(GetDlgItem(hDlg,IDC_PHRASE1))+1;

					gpp->pszPassPhrase = (char *)secAlloc (gpp->context, i);

					if (gpp->pszPassPhrase) 
					{
						SEGetText(GetDlgItem(hDlg,IDC_PHRASE1),gpp->pszPassPhrase);

						if(PassphraseLengthAndQualityOK(hDlg,options,gpp->pszPassPhrase))
						{
							ClearPassphrases(hDlg,gpp);
							SaveOptionSettings(gpp->hwndOptionsControl);
							EndDialog (hDlg, kPGPError_NoErr);
							break;
						}
						// Passphrase quality not enough
						ClearPassphrases(hDlg,gpp);
						FreePassphrases(gpp);
						break;
					}
						
					// Couldn't allocate passphrases
					ClearPassphrases(hDlg,gpp);
					FreePassphrases(gpp);
					EndDialog (hDlg, kPGPError_OutOfMemory);
					break;
				}
			}
			break;
		}
	}
	return FALSE;
}

// Just a simple decryption
	PGPError
pgpPassphraseDialogPlatform(
	PGPContextRef					context,
	CPGPPassphraseDialogOptions 	*options)
{
	if (options->mTextUI) {
		return pgpPassphraseCL(context, options);
	} else {
		PGPError err;
		GPP	gpp;
		memset(&gpp,0x00,sizeof(GPP));
		gpp.context=context;
		gpp.options=options;
		
		InitRandomKeyHook(&hhookKeyboard, &hhookMouse);
		InstallSecurityHooks();
		
		err = DialogBoxParam (gPGPsdkUILibInst, 
							  MAKEINTRESOURCE (IDD_PASSPHRASE), 
							  options->mHwndParent,
							  (DLGPROC)pgpPassphraseDlgProc, (LPARAM)&gpp);
		
		*(options->mPassphrasePtr)=gpp.pszPassPhrase;
		
		UninstallSecurityHooks();
		UninitRandomKeyHook(hhookKeyboard, hhookMouse);
		
		return(err);
	}
}

//	________________________
//
//  Dialog Message procedure

static PGPBoolean Working=FALSE;

BOOL CALLBACK 
pgpDecryptionPassphraseDlgProc (
		HWND	hDlg, 
		UINT	uMsg, 
		WPARAM	wParam,
		LPARAM	lParam) 
{
	CPGPDecryptionPassphraseDialogOptions *options;
	GPP				*gpp;
	INT				i;
	DWORD			Common;

	Common=DoCommonCalls (hDlg,uMsg,wParam,lParam); 

	if(Common)
		return Common;

	switch (uMsg) 
	{
		case WM_INITDIALOG:
		{
			gpp=(GPP *)GetWindowLong (hDlg, GWL_USERDATA);
			options = (CPGPDecryptionPassphraseDialogOptions *)
				gpp->options;

			gpp->iNextTabControl = IDOK;

			gpp->bNoOK=!InitEncryptedToKeyListBox (hDlg, options);

			SEInit(gpp->context,GetDlgItem(hDlg, IDC_PHRASE1),TRUE,
				SpaceEditCallback,gpp);

			gpp->hwndOptionsControl=DisplayOptions(hDlg,options->mDialogOptions,200);

			return FALSE;
		}


		case WM_COMMAND:
		{
			gpp=(GPP *)GetWindowLong (hDlg, GWL_USERDATA);
			options = (CPGPDecryptionPassphraseDialogOptions *)
				gpp->options;

			switch(LOWORD (wParam)) 
			{

				case IDC_KEYLISTBOX:
				{
					if(HIWORD(wParam)==LBN_DBLCLK)
					{
						if(!Working)
						{
							Working=TRUE;

							if(IsntNull(options->mMissingKeyIDList))
							{
								PGPError	err;
								PGPBoolean	haveNewKeys;

								err=options->SearchForMissingKeys( hDlg, &haveNewKeys );

								PGPsdkUIErrorBox (hDlg,err);
								
								if( haveNewKeys )
								{
									gpp->bNoOK=!InitEncryptedToKeyListBox (hDlg, options);
								}
							}
					
							Working=FALSE;
						}
					}
					break;
				}

				case IDOK: 
				{
					FreePassphrases(gpp);

					i=SEGetTextLength(GetDlgItem(hDlg,IDC_PHRASE1))+1;

					gpp->pszPassPhrase = (char *)secAlloc (gpp->context, i);

					if (gpp->pszPassPhrase) 
					{
						CHAR sz[256];

						SEGetText(GetDlgItem(hDlg,IDC_PHRASE1),gpp->pszPassPhrase);

						if(PassphraseLengthAndQualityOK(hDlg,options,gpp->pszPassPhrase))
						{
							if (!options->mFindMatchingKey)
							{
								ClearPassphrases(hDlg,gpp);
								SaveOptionSettings(gpp->hwndOptionsControl);
								EndDialog (hDlg, kPGPError_NoErr);
								break;
							}

							*(options->mPassphraseKeyPtr)=
								GetKeyForPassphrase(options->mKeySet,
									gpp->pszPassPhrase,FALSE);
	
							if (*(options->mPassphraseKeyPtr)!=NULL)
							{
								ClearPassphrases(hDlg,gpp);
								SaveOptionSettings(gpp->hwndOptionsControl);
								EndDialog (hDlg, kPGPError_NoErr);
								break;
							}
							else
							{
//								PGPsdkUIMessageBox (hDlg, IDS_PGPERROR, 
//									IDS_BADDECRYPTPHRASE, MB_OK|MB_ICONEXCLAMATION);
								DoBadPassphraseShake(hDlg,gpp);

								LoadString (gPGPsdkUILibInst, IDS_WRONGDECRYPTPHRASE, 
									sz, sizeof (sz));
								SetDlgItemText (hDlg, IDC_PROMPTSTRING, sz);
								IBSetHighlightMode(GetDlgItem(hDlg,IDC_PROMPTSTRING),
									FALSE);
							}
						}
						// Bad passphrase/quality
						ClearPassphrases(hDlg,gpp);
						FreePassphrases(gpp);
						break;
					}
			
					// Couldn't allocate passphrases
					ClearPassphrases(hDlg,gpp);
					FreePassphrases(gpp);
					EndDialog (hDlg, kPGPError_OutOfMemory);
					break;
				}
			}
			break;
		}
	}
	return FALSE;
}

// Show the recipients
	PGPError
pgpDecryptionPassphraseDialogPlatform(
	PGPContextRef							context,
	CPGPDecryptionPassphraseDialogOptions	*options)
{
	PGPError err;
	GPP	gpp;

	memset(&gpp,0x00,sizeof(GPP));
	gpp.context=context;
	gpp.options=options;

	InitRandomKeyHook(&hhookKeyboard, &hhookMouse);
	InstallSecurityHooks();

	err = DialogBoxParam (gPGPsdkUILibInst, 
		MAKEINTRESOURCE (IDD_PASSPHRASEDECRYPTKEYS), 
		options->mHwndParent,
		(DLGPROC)pgpDecryptionPassphraseDlgProc, (LPARAM)&gpp);

	*(options->mPassphrasePtr)=gpp.pszPassPhrase;

	UninstallSecurityHooks();
	UninitRandomKeyHook(hhookKeyboard, hhookMouse);

	return(err);
}


//	_________________________________________________________
//
//  Dialog Message procedure - get passphrase and signing key

BOOL CALLBACK 
pgpSigningPassphraseDlgProc (
		HWND	hDlg, 
		UINT	uMsg, 
		WPARAM	wParam,
		LPARAM	lParam) 
{
	CPGPSigningPassphraseDialogOptions *options;
	GPP				*gpp;
	INT				i;
	DWORD			Common;

	Common=DoCommonCalls (hDlg,uMsg,wParam,lParam); 

	if(Common)
		return Common;

	switch (uMsg) 
	{
		case WM_INITDIALOG:
		{
			gpp=(GPP *)GetWindowLong (hDlg, GWL_USERDATA);
			options = (CPGPSigningPassphraseDialogOptions *)
				gpp->options;

			gpp->iNextTabControl = IDOK;

			SEInit(gpp->context,GetDlgItem(hDlg, IDC_PHRASE1),TRUE,
				SpaceEditCallback,gpp);

			if (InitSigningKeyComboBox (hDlg, options)) 
			{
				EnablePassPhraseControl (hDlg, gpp->context);
			}
			else 
			{
				EndDialog (hDlg, kPGPError_UserAbort);//kPGPError_Win32_NoSecretKeys);
			}

			gpp->hwndOptionsControl=DisplayOptions(hDlg,options->mDialogOptions,130);

			return FALSE;
		}

		case WM_COMMAND:
		{
			gpp=(GPP *)GetWindowLong (hDlg, GWL_USERDATA);
			options = (CPGPSigningPassphraseDialogOptions *)
				gpp->options;

			switch (LOWORD (wParam)) 
			{
				case IDOK:
				{
					int ComboSelection;

					FreePassphrases(gpp);

					ComboSelection = SendDlgItemMessage (hDlg, IDC_SIGNKEYCOMBO, 
						CB_GETCURSEL, 0, 0);

					if (ComboSelection == CB_ERR) 
					{
						PGPsdkUIMessageBox (hDlg, 
							IDS_PGPERROR, IDS_NOSIGNKEY, 
							MB_OK | MB_ICONEXCLAMATION);
						break;
					}

					i=SEGetTextLength(GetDlgItem(hDlg,IDC_PHRASE1))+1;

					gpp->pszPassPhrase = (char *)secAlloc (gpp->context, i);

					if (gpp->pszPassPhrase) 
					{
						PGPKeyDBObjRef	key;
						PGPError	err;
						PGPBoolean	bShared;

						SEGetText(GetDlgItem(hDlg,IDC_PHRASE1),gpp->pszPassPhrase);
							
						key = (PGPKeyDBObjRef)SendDlgItemMessage (hDlg, 
							IDC_SIGNKEYCOMBO, CB_GETITEMDATA, ComboSelection, 0);

						err=PGPGetKeyDBObjBooleanProperty( key, kPGPKeyProperty_IsSecretShared, &bShared);
	
						if(IsntPGPError(err) && bShared)
						{
							// So, they want to do a shared key....
							ClearPassphrases(hDlg,gpp);
							FreePassphrases(gpp);
							*(options->mPassphraseKeyPtr) = key;
							EndDialog (hDlg, kPGPError_KeyUnusableForSignature);
							break;
						}

						if(PassphraseLengthAndQualityOK(hDlg,options,gpp->pszPassPhrase))
						{
							if (!options->mVerifyPassphrase)
							{
								ClearPassphrases(hDlg,gpp);
								SaveOptionSettings(gpp->hwndOptionsControl);
								EndDialog (hDlg, kPGPError_NoErr);
								break;
							}

							err=ValidateSigningPhrase(hDlg,gpp,gpp->pszPassPhrase,key);

							if(IsntPGPError(err))
							{
								ClearPassphrases(hDlg,gpp);
								SaveOptionSettings(gpp->hwndOptionsControl);
								EndDialog (hDlg, kPGPError_NoErr);
								break;
							}
						}

						// Bad passphrase/quality
						ClearPassphrases(hDlg,gpp);
						FreePassphrases(gpp);
						break;
					}
					

					// Couldn't allocate passphrases
					ClearPassphrases(hDlg,gpp);
					FreePassphrases(gpp);
					EndDialog (hDlg, kPGPError_OutOfMemory);
					break;
				}

				case IDC_SIGNKEYCOMBO :
				{
					if(HIWORD(wParam)==CBN_SELCHANGE)
						EnablePassPhraseControl (hDlg, gpp->context);
					break;
				}
			}
		}
		break;
	}
	return FALSE;
}

// Signer combo box
	PGPError
pgpSigningPassphraseDialogPlatform(
	PGPContextRef						context,
	CPGPSigningPassphraseDialogOptions 	*options)
{
	// check for UI option if not set the assume Graphical
	// else use command line UI
	if (options->mTextUI) {
		return pgpSigningPassphraseCL(context, options);
	} else {
		PGPError err;
		GPP	gpp;
		
		memset(&gpp,0x00,sizeof(GPP));
		gpp.context=context;
		gpp.options=options;

		InitRandomKeyHook(&hhookKeyboard, &hhookMouse);
		InstallSecurityHooks();
		
		err = DialogBoxParam(gPGPsdkUILibInst, 
							 MAKEINTRESOURCE (IDD_PASSPHRASEOPTIONS), 
							 options->mHwndParent,
							 (DLGPROC)pgpSigningPassphraseDlgProc,
							 (LPARAM)&gpp);
		
		*(options->mPassphrasePtr)=gpp.pszPassPhrase;
		
		UninstallSecurityHooks();
		UninitRandomKeyHook(hhookKeyboard, hhookMouse);
		return(err);
	}
}

//	________________________
//
//  Dialog Message procedure

BOOL CALLBACK 
pgpConfirmationDlgProc (
		HWND	hDlg, 
		UINT	uMsg, 
		WPARAM	wParam,
		LPARAM	lParam) 
{
	CPGPConfirmationPassphraseDialogOptions *options;
	GPP				*gpp;
	INT				i;
	DWORD			Common;

	Common=DoCommonCalls (hDlg,uMsg,wParam,lParam); 

	if(Common)
		return Common;

	switch (uMsg) 
	{
		case WM_INITDIALOG:
		{
			gpp=(GPP *)GetWindowLong (hDlg, GWL_USERDATA);
			options = (CPGPConfirmationPassphraseDialogOptions *)
				gpp->options;

			gpp->iNextTabControl = IDC_PHRASE2;
			gpp->hwndQuality = GetDlgItem (hDlg, IDC_PHRASEQUALITY);
			gpp->hwndMinQuality = GetDlgItem (hDlg, IDC_MINQUALITY);

			SEInit(gpp->context,GetDlgItem(hDlg, IDC_PHRASE1),TRUE,
				SpaceEditCallback,gpp);

			SEInit(gpp->context,GetDlgItem(hDlg, IDC_PHRASE2),TRUE,
				SpaceEditCallback,gpp);

			gpp->hwndOptionsControl=DisplayOptions(hDlg,options->mDialogOptions,215);

			// If we aren't showing quality, don't show either bar
			if(!options->mShowPassphraseQuality)
			{
				EnableWindow (GetDlgItem (hDlg, IDC_STATICPASSQUAL), FALSE);
				ShowWindow (GetDlgItem (hDlg, IDC_STATICPASSQUAL), SW_HIDE);
				EnableWindow (GetDlgItem (hDlg, IDC_STATICMINQUAL), FALSE);
				ShowWindow (GetDlgItem (hDlg, IDC_STATICMINQUAL), SW_HIDE);

				EnableWindow (GetDlgItem (hDlg, IDC_MINQUALITY), FALSE);
				ShowWindow (GetDlgItem (hDlg, IDC_MINQUALITY), SW_HIDE);
				EnableWindow (GetDlgItem (hDlg, IDC_PHRASEQUALITY), FALSE);
				ShowWindow (GetDlgItem (hDlg, IDC_PHRASEQUALITY), SW_HIDE);
			}

			// If we don't give a min quality, don't show bar
			if(options->mMinPassphraseQuality==0)
			{
				EnableWindow (GetDlgItem (hDlg, IDC_STATICMINQUAL), FALSE);
				ShowWindow (GetDlgItem (hDlg, IDC_STATICMINQUAL), SW_HIDE);

				EnableWindow (GetDlgItem (hDlg, IDC_MINQUALITY), FALSE);
				ShowWindow (GetDlgItem (hDlg, IDC_MINQUALITY), SW_HIDE);
			}

			// Set our min quality benchmark
			SendMessage (gpp->hwndMinQuality, PBM_SETPOS, 
				options->mMinPassphraseQuality, 0);

			// Disable OK till we get a good passphrase if mins are active
			if(options->mMinPassphraseQuality!=0)
//			  (options->mMinPassphraseLength!=0))
				EnableWindow (GetDlgItem (hDlg, IDOK), FALSE);

			return FALSE;
		}

		case WM_COMMAND:
		{
			gpp=(GPP *)GetWindowLong (hDlg, GWL_USERDATA);
			options = (CPGPConfirmationPassphraseDialogOptions *)
				gpp->options;

			switch(LOWORD (wParam)) 
			{
				case IDOK: 
				{
					FreePassphrases(gpp);

					i=SEGetTextLength(GetDlgItem(hDlg,IDC_PHRASE1))+1;

					gpp->pszPassPhrase = (char *)secAlloc (gpp->context, i);

					if (gpp->pszPassPhrase) 
					{
						SEGetText(GetDlgItem(hDlg,IDC_PHRASE1),gpp->pszPassPhrase);

						i=SEGetTextLength(GetDlgItem(hDlg,IDC_PHRASE2))+1;

						gpp->pszPassPhraseConf = (char *)secAlloc (gpp->context, i);
						if (gpp->pszPassPhraseConf) 
						{
							SEGetText(GetDlgItem(hDlg,IDC_PHRASE2),gpp->pszPassPhraseConf);

							if(PassphraseLengthAndQualityOK(hDlg,options,gpp->pszPassPhrase))
							{
								if (!lstrcmp (gpp->pszPassPhrase, gpp->pszPassPhraseConf))
								{
									ClearPassphrases(hDlg,gpp);
									if(gpp->pszPassPhraseConf)
									{
										secFree(gpp->pszPassPhraseConf);
										gpp->pszPassPhraseConf=NULL;
									}
									SaveOptionSettings(gpp->hwndOptionsControl);
									EndDialog (hDlg, kPGPError_NoErr);
									break;
								}
								else
								{
//									PGPsdkUIMessageBox (hDlg, IDS_PGPERROR, 
//										IDS_PHRASEMISMATCH, MB_OK|MB_ICONEXCLAMATION);
									char sz[100];
									HWND hwndStatic;

									DoBadPassphraseShake(hDlg,gpp);

									hwndStatic=GetDlgItem(hDlg,IDC_PROMPTSTRING);

									LoadString (gPGPsdkUILibInst, IDS_PHRASEMISMATCH, sz, sizeof (sz));
									SetWindowText(hwndStatic,sz);

									IBSetHighlightMode(hwndStatic,FALSE);
								}
							}
							// Badpassphrase/quality
							ClearPassphrases(hDlg,gpp);
							FreePassphrases(gpp);
							SendMessage (gpp->hwndQuality, PBM_SETPOS, 0, 0);
							if(options->mMinPassphraseQuality!=0)
								EnableWindow (GetDlgItem (hDlg, IDOK), FALSE);
							break;
						}
					}
					
					// Couldn't allocate passphrases
					ClearPassphrases(hDlg,gpp);
					FreePassphrases(gpp);
					EndDialog (hDlg, kPGPError_OutOfMemory);
					break;
				}
			}
			break;
		}
	}
	return FALSE;
}

// Double edit window
	PGPError
pgpConfirmationPassphraseDialogPlatform(
	PGPContextRef								context,
	CPGPConfirmationPassphraseDialogOptions 	*options)
{
	PGPError err;
	GPP	gpp;

	if (options->mTextUI) {
		return pgpConfirmationPassphraseDialogCL(context, options);
	}
	else {
		memset(&gpp,0x00,sizeof(GPP));
		gpp.context=context;
		gpp.options=options;

		InitRandomKeyHook(&hhookKeyboard, &hhookMouse);
		InstallSecurityHooks();
		
		err = DialogBoxParam (gPGPsdkUILibInst, 
			MAKEINTRESOURCE (IDD_PASSPHRASEENCRYPT),
			options->mHwndParent,
			(DLGPROC)pgpConfirmationDlgProc, (LPARAM)&gpp);

		*(options->mPassphrasePtr)=gpp.pszPassPhrase;

		UninstallSecurityHooks();
		UninitRandomKeyHook(hhookKeyboard, hhookMouse);

		return(err);
	}
}

//	_________________________________________________________
//
//  Dialog Message procedure - get passphrase and signing key

BOOL CALLBACK 
pgpKeyPassphraseDlgProc (
		HWND	hDlg, 
		UINT	uMsg, 
		WPARAM	wParam,
		LPARAM	lParam) 
{
	CPGPKeyPassphraseDialogOptions *options;
	GPP				*gpp;
	INT				i;
	DWORD			Common;

	Common=DoCommonCalls (hDlg,uMsg,wParam,lParam); 

	if(Common)
		return Common;

	switch (uMsg) 
	{
		case WM_INITDIALOG:
		{
			RECT rc;
			int iTextWidth;
			HDC hdc;
			char szNameFinal[kPGPMaxUserIDSize];

			gpp=(GPP *)GetWindowLong (hDlg, GWL_USERDATA);
			options = (CPGPKeyPassphraseDialogOptions *)
				gpp->options;

			gpp->iNextTabControl = IDOK;

			SEInit(gpp->context,GetDlgItem(hDlg, IDC_PHRASE1),TRUE,
				SpaceEditCallback,gpp);

			GetClientRect(GetDlgItem(hDlg,IDC_KEYNAME), &rc);

			iTextWidth = rc.right-rc.left;
			hdc = GetDC (GetDlgItem(hDlg,IDC_KEYNAME));
			GetKeyString(hdc,iTextWidth,options->mDefaultKey,szNameFinal);
			SetWindowText(GetDlgItem(hDlg,IDC_KEYNAME),szNameFinal);
			ReleaseDC (GetDlgItem (hDlg, IDC_KEYNAME), hdc);

			gpp->hwndOptionsControl=DisplayOptions(hDlg,options->mDialogOptions,130);

			return FALSE;
		}

		case WM_COMMAND:
		{
			gpp=(GPP *)GetWindowLong (hDlg, GWL_USERDATA);
			options = (CPGPKeyPassphraseDialogOptions *)
				gpp->options;

			switch (LOWORD (wParam)) 
			{
				case IDOK:
				{
					PGPBoolean PassValid = FALSE;

					FreePassphrases(gpp);

					i=SEGetTextLength(GetDlgItem(hDlg,IDC_PHRASE1))+1;

					gpp->pszPassPhrase = (char *)secAlloc (gpp->context, i);

					if (gpp->pszPassPhrase) 
					{
						SEGetText(GetDlgItem(hDlg,IDC_PHRASE1),gpp->pszPassPhrase);
			
						if(PassphraseLengthAndQualityOK(hDlg,options,gpp->pszPassPhrase))
						{
							if(!options->mVerifyPassphrase)
							{
								ClearPassphrases(hDlg,gpp);
								SaveOptionSettings(gpp->hwndOptionsControl);
								EndDialog (hDlg, kPGPError_NoErr);
								break;
							}

							// Since we previously checked whether this key actually
							// needs a passphrase with the kPGPKeyPropNeedsPassphrase
							// property, we can now reject empty strings out-of-hand.
							// This prevents the problem of permitting an empty
							// string as a valid phrase when the passphrase is cached.
							if (i > 1)
							{
								PassValid=PGPPassphraseIsValid (options->mDefaultKey, 
									PGPOPassphrase (gpp->context, gpp->pszPassPhrase),
										options->mCache ?
											PGPOCachePassphrase(gpp->context,
												options->mCacheTimeout, 
												options->mCacheGlobal ) :
											PGPONullOption( gpp->context ),
									PGPOLastOption (gpp->context));
							}

							if(PassValid)
							{
								ClearPassphrases(hDlg,gpp);
								SaveOptionSettings(gpp->hwndOptionsControl);
								EndDialog (hDlg, kPGPError_NoErr);
								break;
							}
							else
							{
//								PGPsdkUIMessageBox (hDlg,
//									IDS_PGPERROR,IDS_BADPASSREENTER,
//									MB_OK|MB_ICONSTOP);
								char sz[100];
								HWND hwndStatic;

								DoBadPassphraseShake(hDlg,gpp);

								hwndStatic=GetDlgItem(hDlg,IDC_PROMPTSTRING);

								LoadString (gPGPsdkUILibInst, IDS_WRONGDECRYPTPHRASE, sz, sizeof (sz));
								SetWindowText(hwndStatic,sz);

								IBSetHighlightMode(hwndStatic,FALSE);
							}
						}
						// Bad passphrase/quality
						ClearPassphrases(hDlg,gpp);
						FreePassphrases(gpp);
						break;
					}
					
					// Couldn't allocate passphrases
					ClearPassphrases(hDlg,gpp);
					FreePassphrases(gpp);
					EndDialog (hDlg, kPGPError_OutOfMemory);
					break;
				}
			}
		}
		break;
	}
	return FALSE;
}


	PGPError
pgpKeyPassphraseDialogPlatform(
	PGPContextRef					context,
	CPGPKeyPassphraseDialogOptions 	*options)
{
	PGPError err;
	PGPBoolean bShared,bNeedsPhrase;
	
	err=PGPGetKeyDBObjBooleanProperty( options->mDefaultKey,
						  kPGPKeyProperty_IsSecretShared,
						  &bShared);
	
	if(IsntPGPError(err) && bShared)
	{
		// So, they want to do a shared key....
		return(kPGPError_KeyUnusableForSignature);
	}
		
	err=PGPGetKeyDBObjBooleanProperty (options->mDefaultKey,
						  kPGPKeyProperty_NeedsPassphrase,
						  &bNeedsPhrase);
	
	if(IsntPGPError(err) && !bNeedsPhrase)
	{
		*(options->mPassphrasePtr)=(char *)secAlloc (context, 1);
		if(*(options->mPassphrasePtr)==0)
			return(kPGPError_OutOfMemory);
		
		// Doesn't need a passphrase
		strcpy(*(options->mPassphrasePtr),"");
		return(kPGPError_NoErr);
	}
		
	if (options->mTextUI) {
		return pgpKeyPassphraseCL(context, options);
	} else {
		GPP	gpp;
		
		memset(&gpp,0x00,sizeof(GPP));
		gpp.context=context;
		gpp.options=options;
		
		InitRandomKeyHook(&hhookKeyboard, &hhookMouse);
		InstallSecurityHooks();
		
		err = DialogBoxParam (gPGPsdkUILibInst, 
							  MAKEINTRESOURCE (IDD_KEYPASSPHRASE), 
							  options->mHwndParent,
							  (DLGPROC)pgpKeyPassphraseDlgProc, (LPARAM)&gpp);
		
		*(options->mPassphrasePtr)=gpp.pszPassPhrase;
		
		UninstallSecurityHooks();
		UninitRandomKeyHook(hhookKeyboard, hhookMouse);
		
		return(err);
	}
}
